const leadInput = document.getElementById("leadMinutes");
const playSoundInput = document.getElementById("playSound");
const soundChoiceSelect = document.getElementById("soundChoice");
const soundControls = document.getElementById("soundControls");
const previewSoundBtn = document.getElementById("previewSound");
const statusEl = document.getElementById("status");
const saveBtn = document.getElementById("saveBtn");
const testBtn = document.getElementById("testBtn");

if (
  !leadInput ||
  !playSoundInput ||
  !soundChoiceSelect ||
  !soundControls ||
  !previewSoundBtn ||
  !statusEl ||
  !saveBtn ||
  !testBtn
) {
  console.error("Options page missing required elements");
  // Avoid runtime errors if the DOM isn't what we expect
  throw new Error("Missing required options page elements");
}

// ---------- Helpers ----------

function clampLeadMinutes(value) {
  if (!Number.isFinite(value)) return 10;
  return Math.max(1, Math.min(180, Math.round(value)));
}

function getCurrentLeadMinutes() {
  // leadInput.value is a string; Number("") => 0, Number("abc") => NaN
  const n = Number(leadInput.value);
  return clampLeadMinutes(n);
}

function setStatus(text, ms = 1500) {
  statusEl.textContent = text;
  if (ms > 0) {
    setTimeout(() => {
      if (statusEl.textContent === text) statusEl.textContent = "";
    }, ms);
  }
}

function setSaveEnabled(enabled) {
  saveBtn.disabled = !enabled;

  if (enabled) {
    saveBtn.style.background = "#1a73e8";
    saveBtn.style.color = "#fff";
    saveBtn.style.border = "1px solid #1a73e8";
    saveBtn.style.cursor = "pointer";
  } else {
    saveBtn.style.background = "";
    saveBtn.style.color = "";
    saveBtn.style.border = "";
    saveBtn.style.cursor = "default";
  }
}

// ---------- State ----------

let lastSavedLeadMinutes = 10;
let lastSavedPlaySound = true;
let lastSavedSoundChoice = "ping";

// ---------- Dirty checking ----------

function isDirty() {
  return (
    getCurrentLeadMinutes() !== lastSavedLeadMinutes ||
    playSoundInput.checked !== lastSavedPlaySound ||
    soundChoiceSelect.value !== lastSavedSoundChoice
  );
}

function checkDirty() {
  const dirty = isDirty();
  setSaveEnabled(dirty);

  if (dirty) {
    setStatus("Unsaved changes", 0);
  } else if (statusEl.textContent === "Unsaved changes") {
    statusEl.textContent = "";
  }
}

function updateSoundControlsVisibility() {
  soundControls.classList.toggle("hidden", !playSoundInput.checked);
}

// ---------- Load ----------

async function load() {
  try {
    const { leadMinutes = 10, playSound = true, soundChoice = "ping" } = await chrome.storage.sync.get([
      "leadMinutes",
      "playSound",
      "soundChoice",
    ]);
    lastSavedLeadMinutes = clampLeadMinutes(Number(leadMinutes));
    leadInput.value = String(lastSavedLeadMinutes);
    lastSavedPlaySound = Boolean(playSound);
    playSoundInput.checked = lastSavedPlaySound;
    lastSavedSoundChoice = soundChoice || "ping";
    soundChoiceSelect.value = lastSavedSoundChoice;
    updateSoundControlsVisibility();

    setSaveEnabled(false);
    setStatus(`Loaded: ${lastSavedLeadMinutes} minutes`, 1200);
  } catch (e) {
    console.error("Failed to load settings:", e);
    setStatus("Failed to load settings", 2000);
    setSaveEnabled(false);
  }
}

// ---------- Save ----------

async function save() {
  try {
    setSaveEnabled(false);

    const leadMinutes = getCurrentLeadMinutes();
    const playSound = Boolean(playSoundInput.checked);
    const soundChoice = soundChoiceSelect.value || "ping";

    // Normalize displayed value
    leadInput.value = String(leadMinutes);

    await chrome.storage.sync.set({ leadMinutes, playSound, soundChoice });

    lastSavedLeadMinutes = leadMinutes;
    lastSavedPlaySound = playSound;
    lastSavedSoundChoice = soundChoice;
    setStatus(`Saved: ${leadMinutes} minutes`);
    checkDirty(); // ensure button state matches
  } catch (e) {
    console.error("Failed to save settings:", e);
    setStatus("Failed to save settings", 2000);
    // Re-enable only if still dirty
    setSaveEnabled(isDirty());
  }
}

// ---------- Test reminder ----------

async function testReminderNow() {
  try {
    testBtn.disabled = true;
    setStatus("Triggering test reminder…", 1200);

    await chrome.runtime.sendMessage({ type: "TEST_REMINDER_NOW" });
    setStatus("Test reminder sent (check for a toast).", 1800);
  } catch (e) {
    console.error("Failed to trigger test reminder:", e);
    setStatus("Failed to trigger test reminder", 2000);
  } finally {
    testBtn.disabled = false;
  }
}

// ---------- Events ----------

saveBtn.addEventListener("click", save);
testBtn.addEventListener("click", testReminderNow);

leadInput.addEventListener("input", checkDirty);
playSoundInput.addEventListener("change", () => {
  checkDirty();
  updateSoundControlsVisibility();
});
soundChoiceSelect.addEventListener("change", checkDirty);
previewSoundBtn.addEventListener("click", async () => {
  const soundChoice = soundChoiceSelect.value || "ping";
  try {
    await chrome.runtime.sendMessage({ type: "PREVIEW_SOUND", options: { soundChoice } });
  } catch (e) {
    console.error("Failed to preview sound:", e);
  }
});

leadInput.addEventListener("keydown", (e) => {
  if (e.key === "Enter") {
    e.preventDefault();
    if (!saveBtn.disabled) save();
  } else if (e.key === "Escape") {
    leadInput.value = String(lastSavedLeadMinutes);
    checkDirty();
  }
});

// Prevent accidental scroll-wheel changes
leadInput.addEventListener("wheel", () => leadInput.blur(), { passive: true });

// Optional: clamp on blur (keeps the field tidy without forcing mid-typing changes)
// leadInput.addEventListener("blur", () => {
//   leadInput.value = String(getCurrentLeadMinutes());
//   checkDirty();
// });

// ---------- Init ----------
load();
