/**
 * Meeting Radar — MV3 Service Worker
 * - Primary calendar only
 * - Ignores all-day events + declined events
 * - Polls every minute while Chrome is running
 * - Hybrid UX: OS toast notification -> click or "Snooze…" opens reminder window
 * - Queue: handles back-to-back meetings (one toast at a time)
 * - Instance identity: iCalUID + originalStartTime
 * - Join link detection: hangoutLink + conferenceData + description/location URL scan + provider preference
 * - Join link previews: "Google Meet — abc-defg-hij", etc. (label used as link text in reminder UI)
 * - Storage cleanup: expires snoozes + dismiss markers + old queue entries + "recentlySeen" map
 * - Badge countdown: minutes until next upcoming meeting (within lookahead)
 * - Options: supports TEST_REMINDER_NOW message
 *
 * Updates merged:
 * - requireInteraction: true (keeps toast visible until user action, where supported)
 * - Option A: clear toast immediately when user clicks Join toast button
 * - Option B: after meeting start passes, toast message updates to “Meeting started …” (via poll tick)
 * - If user ignores reminder, after 10 min allow next reminder through (INFLIGHT_ADVANCE_MS)
 * - Snooze button: toast is cleared when opening reminder window, so it doesn't linger
 * - Toast update: schedule a one-shot alarm at start time to flip to “Meeting started …”
 */

console.log("Meeting Radar SW loaded", new Date().toISOString());

const CONFIG = {
  ALARM_NAME: "poll_calendar",
  CLEANUP_ALARM: "cleanup_state",
  LOOKAHEAD_HOURS: 12,

  // Queue / toast behavior
  // How long we’ll wait for user action before we allow the next reminder through.
  INFLIGHT_ADVANCE_MS: 10 * 60 * 1000, // 10 minutes
  QUEUE_PAST_GRACE_MS: 30 * 60 * 1000, // drop queued reminders that are >30 min past start

  // Suppression / dedupe behavior
  KEEP_DISMISS_MS: 24 * 60 * 60 * 1000, // keep dismiss markers for 24h
  KEEP_SEEN_MS: 6 * 60 * 60 * 1000, // keep "recentlySeen" for 6h to avoid repeated enqueues

  BADGE_TICK_ALARM: "badge_tick",
  BADGE_TICK_FUZZ_MS: 200, // tiny buffer so we cross the minute boundary

  TOAST_UPDATE_ALARM: "toast_update",
};

// -------------------- Settings --------------------

async function getSettings() {
  const got = await chrome.storage.sync.get(["leadMinutes", "playSound", "soundChoice"]);
  const leadMinutes = typeof got.leadMinutes === "number" ? got.leadMinutes : Number(got.leadMinutes);
  const playSound = typeof got.playSound === "boolean" ? got.playSound : Boolean(got.playSound);
  const soundChoice = typeof got.soundChoice === "string" ? got.soundChoice : "ping";
  return {
    leadMinutes: Number.isFinite(leadMinutes) ? leadMinutes : 10,
    playSound: typeof got.playSound === "undefined" ? true : playSound,
    soundChoice,
  };
}

async function getLocalState(keys) {
  return chrome.storage.local.get(keys);
}

function snoozeAlarmName(eventKey) {
  return `snooze:${encodeURIComponent(eventKey)}`;
}

function decodeSnoozeAlarmName(alarmName) {
  if (!alarmName || !alarmName.startsWith("snooze:")) return "";
  return decodeURIComponent(alarmName.slice("snooze:".length));
}

async function ensureOffscreenAudioDocument() {
  try {
    if (await chrome.offscreen.hasDocument()) return;
    await chrome.offscreen.createDocument({
      url: "offscreen.html",
      reasons: ["AUDIO_PLAYBACK"],
      justification: "Play a short sound when a reminder toast is shown.",
    });
  } catch (_) {
    // Ignore if offscreen is unavailable or already created.
  }
}

async function playToastSound() {
  try {
    const { playSound, soundChoice } = await getSettings();
    if (!playSound) return;
    await ensureOffscreenAudioDocument();
    await chrome.runtime.sendMessage({ type: "PLAY_SOUND", options: { soundChoice } });
  } catch (_) {}
}

// -------------------- Event filtering --------------------

function isAllDay(ev) {
  return !!(ev && ev.start && ev.start.date); // all-day uses "date" not "dateTime"
}

function getSelfResponseStatus(ev) {
  const attendees = ev && ev.attendees ? ev.attendees : [];
  const me = attendees.find((a) => a && a.self);
  return me && me.responseStatus ? me.responseStatus : "accepted"; // if no attendees, treat as accepted
}

function toMs(dateTimeStr) {
  return new Date(dateTimeStr).getTime();
}

function makeEventKey(ev) {
  const iCalUID = ev && ev.iCalUID ? ev.iCalUID : "";
  const originalStart =
    (ev && ev.originalStartTime && ev.originalStartTime.dateTime) ||
    (ev && ev.originalStartTime && ev.originalStartTime.date) ||
    (ev && ev.start && ev.start.dateTime) ||
    (ev && ev.start && ev.start.date) ||
    "";

  if (iCalUID && originalStart) return `${iCalUID}|${originalStart}`;

  const parts = [
    ev && ev.id ? ev.id : "",
    ev && ev.recurringEventId ? ev.recurringEventId : "",
    originalStart,
  ].filter(Boolean);

  return parts.join("|");
}

// -------------------- Badge countdown --------------------

function getActionApi() {
  try {
    if (typeof chrome === "undefined") return null;
    return chrome.action || null;
  } catch (_) {
    return null;
  }
}

function safeActionCall(methodName, args) {
  try {
    const action = getActionApi();
    if (!action) return;
    const fn = action[methodName];
    if (typeof fn !== "function") return;
    fn.call(action, args);
  } catch (_) {
    // never crash service worker for badge
  }
}

function setBadge(text) {
  safeActionCall("setBadgeText", { text: text || "" });
}

function setBadgeStyle() {
  safeActionCall("setBadgeBackgroundColor", { color: "#1a73e8" });
}

function minutesUntil(ms) {
  const diff = ms - Date.now();
  if (diff <= 0) return 0;
  return Math.ceil(diff / 60000);
}

function updateBadgeForNextStart(nextStartMs) {
  if (!nextStartMs || !Number.isFinite(nextStartMs)) {
    setBadge("");
    return;
  }
  const mins = minutesUntil(nextStartMs);
  setBadge(mins > 99 ? "99+" : String(mins));
}

function msUntilNextMinuteBoundary() {
  const now = Date.now();
  const nextBoundary = now - (now % 60000) + 60000;
  return Math.max(0, nextBoundary - now + CONFIG.BADGE_TICK_FUZZ_MS);
}

async function scheduleBadgeTick() {
  // Schedule a one-shot alarm for the next minute boundary
  const when = Date.now() + msUntilNextMinuteBoundary();
  chrome.alarms.create(CONFIG.BADGE_TICK_ALARM, { when });
}

async function setNextUpcomingStartMs(nextStartMs) {
  await chrome.storage.local.set({ nextUpcomingStartMs: nextStartMs ?? null });
}

async function refreshBadgeFromStoredNextStart() {
  const { nextUpcomingStartMs = null } = await chrome.storage.local.get(["nextUpcomingStartMs"]);
  updateBadgeForNextStart(nextUpcomingStartMs);
}


// -------------------- URL extraction + join preview --------------------

function extractUrls(text) {
  if (!text) return [];
  const re = /\bhttps?:\/\/[^\s<>"']+/gi;
  const matches = text.match(re) || [];
  return matches.map((u) => u.replace(/[)\].,;:]+$/g, ""));
}

function scoreJoinUrl(url) {
  const u = String(url || "").toLowerCase();
  if (u.includes("meet.google.com")) return 100;
  if (u.includes("zoom.us") || u.includes("zoom.com")) return 90;
  if (u.includes("teams.microsoft.com") || u.includes("teams.live.com")) return 85;
  if (u.includes("webex.com")) return 80;
  if (u.includes("gotomeeting.com")) return 70;
  if (u.includes("bluejeans.com")) return 65;
  return 10;
}

function pickBestUrl(urls) {
  if (!urls || !urls.length) return "";
  const sorted = urls.slice().sort((a, b) => scoreJoinUrl(b) - scoreJoinUrl(a));
  return sorted[0] || "";
}

function extractJoinUrl(ev) {
  if (ev && ev.hangoutLink) return ev.hangoutLink;

  const eps = ev && ev.conferenceData && ev.conferenceData.entryPoints ? ev.conferenceData.entryPoints : [];
  const video = eps.find((ep) => ep && ep.entryPointType === "video" && ep.uri);
  if (video && video.uri) return video.uri;

  const urls = []
    .concat(extractUrls(ev && ev.description ? ev.description : ""))
    .concat(extractUrls(ev && ev.location ? ev.location : ""));

  return pickBestUrl(urls);
}

function hostOf(url) {
  try {
    return new URL(url).hostname.toLowerCase();
  } catch {
    return "";
  }
}

function describeJoinUrl(url) {
  if (!url) return { provider: "", label: "", url: "" };

  const host = hostOf(url);
  const u = String(url).toLowerCase();

  // Google Meet
  if (host.endsWith("meet.google.com")) {
    let code = "";
    const direct = String(url).match(/meet\.google\.com\/([a-z]{3}-[a-z]{4}-[a-z]{3})/i);
    if (direct && direct[1]) code = direct[1];

    if (!code) {
      try {
        const parts = new URL(url).pathname.split("/").filter(Boolean);
        const last = parts[parts.length - 1];
        if (last && last.length >= 8) code = last;
      } catch (_) {}
    }

    const suffix = code ? ` — ${code}` : "";
    return { provider: "Google Meet", label: `Google Meet${suffix}`, url };
  }

  // Zoom
  if (host.endsWith("zoom.us") || host.endsWith("zoom.com")) {
    const m =
      String(url).match(/\/j\/(\d{9,12})/i) ||
      String(url).match(/[?&]confno=(\d{9,12})/i) ||
      String(url).match(/[?&]meeting_id=(\d{9,12})/i);
    const id = m && m[1] ? ` — ${m[1]}` : "";
    return { provider: "Zoom", label: `Zoom${id}`, url };
  }

  // Microsoft Teams
  if (host.endsWith("teams.microsoft.com") || u.includes("teams.live.com")) {
    return { provider: "Microsoft Teams", label: "Microsoft Teams", url };
  }

  // Webex
  if (host.includes("webex.com")) {
    return { provider: "Webex", label: "Webex", url };
  }

  // GoToMeeting
  if (host.includes("gotomeeting.com")) {
    return { provider: "GoToMeeting", label: "GoToMeeting", url };
  }

  // Generic fallback: domain
  const prettyHost = host ? host.replace(/^www\./, "") : "Join link";
  return { provider: "Join link", label: prettyHost, url };
}

// -------------------- OAuth token --------------------

async function getToken(interactive) {
  return new Promise((resolve, reject) => {
    chrome.identity.getAuthToken({ interactive }, (token) => {
      if (chrome.runtime.lastError || !token) {
        reject(chrome.runtime.lastError || new Error("No auth token"));
      } else {
        resolve(token);
      }
    });
  });
}

async function removeCachedToken(token) {
  return new Promise((resolve) => {
    chrome.identity.removeCachedAuthToken({ token }, () => resolve());
  });
}

// -------------------- Calendar fetch --------------------

async function fetchUpcomingEvents(token) {
  const now = new Date();
  const timeMin = now.toISOString();
  const timeMax = new Date(now.getTime() + CONFIG.LOOKAHEAD_HOURS * 3600 * 1000).toISOString();

  const url = new URL("https://www.googleapis.com/calendar/v3/calendars/primary/events");
  url.searchParams.set("timeMin", timeMin);
  url.searchParams.set("timeMax", timeMax);
  url.searchParams.set("singleEvents", "true");
  url.searchParams.set("orderBy", "startTime");
  url.searchParams.set("maxResults", "50");
  url.searchParams.set("conferenceDataVersion", "1");

  const res = await fetch(url.toString(), {
    headers: { Authorization: `Bearer ${token}` },
  });

  if (!res.ok) {
    const text = await res.text();
    const err = new Error(`Calendar API error: ${res.status} ${text}`);
    err.status = res.status;
    throw err;
  }

  const json = await res.json();
  return json.items || [];
}

async function fetchUpcomingEventsWithRetry(token) {
  try {
    return await fetchUpcomingEvents(token);
  } catch (e) {
    if (e && e.status === 401) {
      await removeCachedToken(token);
      const newToken = await getToken(true);
      return await fetchUpcomingEvents(newToken);
    }
    throw e;
  }
}

// -------------------- Storage suppression --------------------

async function getSuppression(eventKey) {
  const { snoozedUntilByKey = {}, dismissedByKey = {}, snoozedPayloadByKey = {} } = await getLocalState([
    "snoozedUntilByKey",
    "dismissedByKey",
    "snoozedPayloadByKey",
  ]);

  if (dismissedByKey[eventKey]) {
    return { suppressed: true, reason: "dismissed", snoozeUntil: snoozedUntilByKey[eventKey] };
  }

  const snoozeUntil = snoozedUntilByKey[eventKey];
  const snoozedPayload = snoozedPayloadByKey[eventKey] || null;
  if (typeof snoozeUntil === "number" && Date.now() < snoozeUntil) {
    return { suppressed: true, reason: "snoozed", snoozeUntil, snoozedPayload };
  }

  return { suppressed: false, reason: "", snoozeUntil, snoozedPayload };
}

async function markSeen(eventKey) {
  const { recentlySeenByKey = {} } = await getLocalState(["recentlySeenByKey"]);
  recentlySeenByKey[eventKey] = Date.now();
  await chrome.storage.local.set({ recentlySeenByKey });
}

async function wasSeenRecently(eventKey) {
  const { recentlySeenByKey = {} } = await getLocalState(["recentlySeenByKey"]);
  const seenAt = recentlySeenByKey[eventKey];
  return typeof seenAt === "number" && Date.now() - seenAt < CONFIG.KEEP_SEEN_MS;
}

// -------------------- Queue / in-flight toast --------------------

async function getQueueState() {
  const got = await getLocalState(["notificationQueue", "inFlightNotification"]);
  return {
    notificationQueue: got.notificationQueue || [],
    inFlightNotification: got.inFlightNotification || null,
  };
}

async function setQueueState(patch) {
  await chrome.storage.local.set(patch);
}

function dedupeByEventKey(items) {
  const seen = new Set();
  const out = [];
  for (let i = 0; i < items.length; i++) {
    const it = items[i];
    if (!it || !it.eventKey) continue;
    if (seen.has(it.eventKey)) continue;
    seen.add(it.eventKey);
    out.push(it);
  }
  return out;
}

async function enqueuePayload(payload) {
  const state = await getQueueState();
  const next = dedupeByEventKey(state.notificationQueue.concat([payload]));
  await setQueueState({ notificationQueue: next });
}

async function dequeueNextPayload() {
  const state = await getQueueState();
  if (!state.notificationQueue.length) return null;
  const head = state.notificationQueue[0];
  const rest = state.notificationQueue.slice(1);
  await setQueueState({ notificationQueue: rest });
  return head;
}

async function scheduleToastUpdateAtStart(startMs) {
  if (!Number.isFinite(startMs)) return;
  const when = Math.max(Date.now() + 1, startMs);
  chrome.alarms.create(CONFIG.TOAST_UPDATE_ALARM, { when });
}

async function scheduleSnoozeAlarm(eventKey, snoozeUntilMs) {
  if (!eventKey || !Number.isFinite(snoozeUntilMs)) return;
  const when = Math.max(Date.now() + 1, snoozeUntilMs);
  chrome.alarms.create(snoozeAlarmName(eventKey), { when });
}

async function clearSnoozeAlarm(eventKey) {
  if (!eventKey) return;
  try {
    await chrome.alarms.clear(snoozeAlarmName(eventKey));
  } catch (_) {}
}

async function clearToastUpdateAlarm() {
  try {
    await chrome.alarms.clear(CONFIG.TOAST_UPDATE_ALARM);
  } catch (_) {}
}

/**
 * Build toast options.
 * Chrome notifications don't "count down" unless we update them.
 * We use poll() (every minute) to flip to "Meeting started" once start passes.
 */
function buildNotificationOptions(payload, mode = "upcoming") {
  const provider = payload?.joinPreview?.provider ? payload.joinPreview.provider : "";
  const joinBtnTitle = payload?.joinUrl ? `Join ${provider}`.trim() : "Join (no link)";
  const base = {
    type: "basic",
    iconUrl: chrome.runtime.getURL("icons/icon128.png"),
    title: payload.title || "Meeting",
    buttons: [{ title: joinBtnTitle }, { title: "Snooze…" }],
    priority: 2,
    requireInteraction: true,
  };

  const timeStr = new Date(payload.startMs).toLocaleTimeString([], {
    hour: "numeric",
    minute: "2-digit",
  });

  if (mode === "started") {
    return {
      ...base,
      message: `Meeting started at ${timeStr}`,
    };
  }

  const mins = minutesUntil(payload.startMs);
  return {
    ...base,
    message: `Starts at ${timeStr} (in ${mins} min)`,
  };
}

async function refreshInFlightNotificationIfNeeded() {
  const { inFlightNotification } = await getQueueState();
  if (!inFlightNotification?.payload || !inFlightNotification.notificationId) return;

  const { notificationId, payload } = inFlightNotification;

  const mode = Date.now() >= payload.startMs ? "started" : "upcoming";
  try {
    await chrome.notifications.update(notificationId, buildNotificationOptions(payload, mode));
  } catch (_) {}
}

async function clearStuckInFlightIfNeeded() {
  const { inFlightNotification } = await getQueueState();
  if (!inFlightNotification) return;

  const createdMs = inFlightNotification.createdMs;
  const notificationId = inFlightNotification.notificationId;

  if (typeof createdMs !== "number") return;

  // If ignored for long enough, clear it and allow the next reminder through
  if (Date.now() - createdMs > CONFIG.INFLIGHT_ADVANCE_MS) {
    // Clear the OS toast to avoid piling up stale notifications
    try {
      if (notificationId) chrome.notifications.clear(notificationId);
    } catch (_) {}

    await clearToastUpdateAlarm();
    await setQueueState({ inFlightNotification: null });
  }
}

async function tryShowNextToast() {
  await clearStuckInFlightIfNeeded();

  const state = await getQueueState();
  if (state.inFlightNotification) return;

  const next = await dequeueNextPayload();
  if (!next) return;

  const notificationId = `event:${next.eventKey}`;
  await setQueueState({
    inFlightNotification: { notificationId, payload: next, createdMs: Date.now() },
  });

  await scheduleToastUpdateAtStart(next.startMs);
  chrome.notifications.create(notificationId, buildNotificationOptions(next, "upcoming"));
  await playToastSound();
}

async function clearInFlightIfMatches(notificationId) {
  const state = await getQueueState();
  const inFlight = state.inFlightNotification;
  if (!inFlight) return false;
  if (inFlight.notificationId !== notificationId) return false;
  await clearToastUpdateAlarm();
  await setQueueState({ inFlightNotification: null });
  return true;
}

// -------------------- Reminder window --------------------

async function openReminderWindow(payload) {
  const payloadB64 = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
  const url = chrome.runtime.getURL(`reminder.html?payload=${encodeURIComponent(payloadB64)}`);

  chrome.windows.create({
    url,
    type: "popup",
    width: 520,
    height: 420,
    focused: true,
  });
}

function buildPayload(eventKey, ev, startMs, joinUrl, joinPreview) {
  return {
    eventKey,
    title: ev?.summary || "(No title)",
    startMs,
    joinUrl,
    joinPreview,
  };
}

// -------------------- Polling logic --------------------

async function poll() {
  const settings = await getSettings();
  const leadMinutes = settings.leadMinutes;

  let token;
  try {
    token = await getToken(false);
  } catch (_) {
    token = await getToken(true);
  }

  const events = await fetchUpcomingEventsWithRetry(token);

  // Badge: minutes until the next upcoming (non-all-day, non-declined) event within lookahead
  const now = Date.now();
  let nextUpcomingStartMs = null;

  const candidateEvents = events.filter((ev) => {
    if (!ev || !ev.start || !ev.start.dateTime) return false;
    if (isAllDay(ev)) return false;
    if (getSelfResponseStatus(ev) === "declined") return false;
    return true;
  });

  for (let i = 0; i < candidateEvents.length; i++) {
    const ev = candidateEvents[i];
    const startMs = toMs(ev.start.dateTime);
    if (startMs < now) continue;

    if (nextUpcomingStartMs === null || startMs < nextUpcomingStartMs) {
      nextUpcomingStartMs = startMs;
    }
  }

  await setNextUpcomingStartMs(nextUpcomingStartMs);
  updateBadgeForNextStart(nextUpcomingStartMs);
  await scheduleBadgeTick();

  // If a toast is visible and meeting has started, update its text
  await refreshInFlightNotificationIfNeeded();

  // Collect all events that should fire now
  const eligiblePayloads = [];

  for (let i = 0; i < candidateEvents.length; i++) {
    const ev = candidateEvents[i];
    const startMs = toMs(ev.start.dateTime);

    const fireFrom = startMs - leadMinutes * 60000;
    const fireTo = startMs;
    const inWindow = now >= fireFrom && now <= fireTo;
    if (!inWindow) continue;

    const eventKey = makeEventKey(ev);

    const suppression = await getSuppression(eventKey);
    if (suppression.suppressed) continue;

    if (await wasSeenRecently(eventKey) && !(suppression.snoozeUntil && now >= suppression.snoozeUntil)) {
      continue;
    }

    const joinUrl = extractJoinUrl(ev);
    const joinPreview = describeJoinUrl(joinUrl);

    eligiblePayloads.push(buildPayload(eventKey, ev, startMs, joinUrl, joinPreview));

    await markSeen(eventKey);
  }

  eligiblePayloads.sort((a, b) => a.startMs - b.startMs);

  for (let i = 0; i < eligiblePayloads.length; i++) {
    await enqueuePayload(eligiblePayloads[i]);
  }

  await tryShowNextToast();
}

// -------------------- Alarms setup (install + startup) --------------------

function ensureAlarms() {
  setBadgeStyle();
  chrome.alarms.create(CONFIG.ALARM_NAME, { periodInMinutes: 1 });
  chrome.alarms.create(CONFIG.CLEANUP_ALARM, { periodInMinutes: 60 });
}

chrome.runtime.onInstalled.addListener(() => {
  ensureAlarms();
});

chrome.runtime.onStartup.addListener(() => {
  ensureAlarms();
});

  chrome.alarms.onAlarm.addListener(async (alarm) => {
    try {
      if (alarm?.name === CONFIG.ALARM_NAME) await poll();
      if (alarm?.name === CONFIG.CLEANUP_ALARM) await cleanupState();

      if (alarm?.name === CONFIG.BADGE_TICK_ALARM) {
        await refreshBadgeFromStoredNextStart();
        await scheduleBadgeTick(); // keep ticking each minute boundary
      }

      if (alarm?.name === CONFIG.TOAST_UPDATE_ALARM) {
        await refreshInFlightNotificationIfNeeded();
      }

      if (alarm?.name && alarm.name.startsWith("snooze:")) {
        const eventKey = decodeSnoozeAlarmName(alarm.name);
        if (!eventKey) return;
        const { snoozedUntilByKey = {}, snoozedPayloadByKey = {} } = await getLocalState([
          "snoozedUntilByKey",
          "snoozedPayloadByKey",
        ]);
        const snoozeUntil = snoozedUntilByKey[eventKey];
        const payload = snoozedPayloadByKey[eventKey];
        if (payload && typeof snoozeUntil === "number" && Date.now() >= snoozeUntil) {
          await enqueuePayload(payload);
          await tryShowNextToast();
          delete snoozedUntilByKey[eventKey];
          delete snoozedPayloadByKey[eventKey];
          await chrome.storage.local.set({ snoozedUntilByKey, snoozedPayloadByKey });
        }
      }
    } catch (_) {}
  });


// -------------------- Toast interactions --------------------

chrome.notifications.onClicked.addListener(async (notificationId) => {
  if (!notificationId || !notificationId.startsWith("event:")) return;

  const state = await getQueueState();
  const inFlight = state.inFlightNotification;
  if (!inFlight || inFlight.notificationId !== notificationId) return;

  // (Optional) keep toast, but since user clicked it we generally clear it
  try {
    chrome.notifications.clear(notificationId);
  } catch (_) {}

  await openReminderWindow(inFlight.payload);
  await clearInFlightIfMatches(notificationId);
  await tryShowNextToast();
});

chrome.notifications.onButtonClicked.addListener(async (notificationId, buttonIndex) => {
  if (!notificationId || !notificationId.startsWith("event:")) return;

  const state = await getQueueState();
  const inFlight = state.inFlightNotification;
  if (!inFlight || inFlight.notificationId !== notificationId) return;

  const payload = inFlight.payload;

  if (buttonIndex === 0) {
    // Join (Option A: always clear toast)
    try {
      chrome.notifications.clear(notificationId);
    } catch (_) {}

    if (payload && payload.joinUrl) {
      // NOTE: requires "tabs" permission in manifest.json
      chrome.tabs.create({ url: payload.joinUrl, active: true }, (tab) => {
        if (tab && tab.windowId !== undefined) {
          try {
            chrome.windows.update(tab.windowId, { focused: true });
          } catch (_) {}
        }
      });
    } else {
      await openReminderWindow(payload);
    }

    await clearInFlightIfMatches(notificationId);
    await tryShowNextToast();
  } else if (buttonIndex === 1) {
    // Snooze... (clear toast so it doesn't linger)
    try {
      chrome.notifications.clear(notificationId);
    } catch (_) {}

    await openReminderWindow(payload);
    await clearInFlightIfMatches(notificationId);
    await tryShowNextToast();
  }
});

chrome.notifications.onClosed.addListener(async (notificationId, byUser) => {
  if (!notificationId || !notificationId.startsWith("event:")) return;

  // Only advance if the user explicitly closed it.
  if (byUser) {
    const cleared = await clearInFlightIfMatches(notificationId);
    if (cleared) await tryShowNextToast();
  }
});

// -------------------- Messages from reminder window / options --------------------

chrome.runtime.onMessage.addListener((msg, _sender, sendResponse) => {
  (async () => {
    if (msg && msg.type === "TEST_REMINDER_NOW") {
      const startMs = Date.now() + 5 * 60000;

      const joinUrl = "https://meet.google.com/abc-defg-hij";
      const joinPreview = describeJoinUrl(joinUrl);

      const payload = {
        eventKey: `test|${startMs}`,
        title: "Test Meeting Reminder",
        startMs,
        joinUrl,
        joinPreview,
      };

      await enqueuePayload(payload);
      await tryShowNextToast();

      // Note: poll will overwrite badge on next tick
      updateBadgeForNextStart(startMs);

      sendResponse({ ok: true });
      return;
    }

    if (msg && msg.type === "PREVIEW_SOUND") {
      await ensureOffscreenAudioDocument();
      await chrome.runtime.sendMessage({ type: "PLAY_SOUND", options: msg.options || {} });
      sendResponse({ ok: true });
      return;
    }

    if (msg && msg.type === "SNOOZE") {
      const got = await getLocalState(["snoozedUntilByKey", "snoozedPayloadByKey"]);
      const snoozedUntilByKey = got.snoozedUntilByKey || {};
      const snoozedPayloadByKey = got.snoozedPayloadByKey || {};
      snoozedUntilByKey[msg.eventKey] = msg.snoozeUntilMs;
      if (msg.payload) snoozedPayloadByKey[msg.eventKey] = msg.payload;
      await chrome.storage.local.set({ snoozedUntilByKey, snoozedPayloadByKey });
      await scheduleSnoozeAlarm(msg.eventKey, msg.snoozeUntilMs);

      // If the reminder window snoozed something that was also in-flight, clear any visible toast.
      const { inFlightNotification } = await getQueueState();
      if (inFlightNotification?.payload?.eventKey === msg.eventKey) {
        try {
          if (inFlightNotification.notificationId) {
            chrome.notifications.clear(inFlightNotification.notificationId);
          }
        } catch (_) {}
        await clearInFlightIfMatches(inFlightNotification.notificationId);
        await tryShowNextToast();
      }

      sendResponse({ ok: true });
      return;
    }

    if (msg && msg.type === "DISMISS") {
      const got = await getLocalState(["dismissedByKey", "snoozedUntilByKey", "snoozedPayloadByKey"]);
      const dismissedByKey = got.dismissedByKey || {};
      const snoozedUntilByKey = got.snoozedUntilByKey || {};
      const snoozedPayloadByKey = got.snoozedPayloadByKey || {};
      dismissedByKey[msg.eventKey] = Date.now(); // timestamp (cleanup expects number)
      delete snoozedUntilByKey[msg.eventKey];
      delete snoozedPayloadByKey[msg.eventKey];
      await chrome.storage.local.set({ dismissedByKey, snoozedUntilByKey, snoozedPayloadByKey });
      await clearSnoozeAlarm(msg.eventKey);
      sendResponse({ ok: true });
      return;
    }

    if (msg && msg.type === "OPEN_JOIN") {
      if (msg.url) {
        // NOTE: requires "tabs" permission in manifest.json
        chrome.tabs.create({ url: msg.url, active: true }, (tab) => {
          if (tab && tab.windowId !== undefined) {
            try {
              chrome.windows.update(tab.windowId, { focused: true });
            } catch (_) {}
          }
        });
      }
      sendResponse({ ok: true });
      return;
    }

    sendResponse({ ok: false });
  })();

  return true; // keep message channel open for async
});

// -------------------- Cleanup --------------------

async function cleanupState() {
  const got = await getLocalState([
    "snoozedUntilByKey",
    "dismissedByKey",
    "notificationQueue",
    "recentlySeenByKey",
    "inFlightNotification",
    "snoozedPayloadByKey",
  ]);

  const snoozedUntilByKey = got.snoozedUntilByKey || {};
  const dismissedByKey = got.dismissedByKey || {};
  const notificationQueue = got.notificationQueue || [];
  const recentlySeenByKey = got.recentlySeenByKey || {};
  const inFlightNotification = got.inFlightNotification || null;
  const snoozedPayloadByKey = got.snoozedPayloadByKey || {};

  const now = Date.now();

  // Remove expired snoozes
  const nextSnoozed = { ...snoozedUntilByKey };
  for (const k of Object.keys(nextSnoozed)) {
    const until = nextSnoozed[k];
    if (typeof until !== "number" || until <= now) delete nextSnoozed[k];
  }
  const nextSnoozedPayloads = { ...snoozedPayloadByKey };
  for (const k of Object.keys(nextSnoozedPayloads)) {
    if (!nextSnoozed[k]) delete nextSnoozedPayloads[k];
  }

  // Remove old dismiss markers
  const nextDismissed = { ...dismissedByKey };
  for (const k of Object.keys(nextDismissed)) {
    const dismissedAt = nextDismissed[k];
    if (typeof dismissedAt !== "number" || dismissedAt + CONFIG.KEEP_DISMISS_MS <= now) delete nextDismissed[k];
  }

  // Remove old "recently seen"
  const nextSeen = { ...recentlySeenByKey };
  for (const k of Object.keys(nextSeen)) {
    const seenAt = nextSeen[k];
    if (typeof seenAt !== "number" || seenAt + CONFIG.KEEP_SEEN_MS <= now) delete nextSeen[k];
  }

  // Clear queued items that are already far in the past
  const nextQueue = dedupeByEventKey(
    notificationQueue.filter(
      (p) => p && typeof p.startMs === "number" && p.startMs + CONFIG.QUEUE_PAST_GRACE_MS > now
    )
  );

  // If we have an in-flight toast that is very old, clear it and advance
  let nextInFlight = inFlightNotification;
  const createdMs = inFlightNotification?.createdMs;

  if (nextInFlight && typeof createdMs === "number" && now - createdMs > CONFIG.INFLIGHT_ADVANCE_MS) {
    try {
      if (nextInFlight.notificationId) chrome.notifications.clear(nextInFlight.notificationId);
    } catch (_) {}
    nextInFlight = null;
  }

  if (!nextInFlight) {
    await clearToastUpdateAlarm();
  }

  for (const k of Object.keys(nextSnoozedPayloads)) {
    if (!nextSnoozed[k]) await clearSnoozeAlarm(k);
  }

  await chrome.storage.local.set({
    snoozedUntilByKey: nextSnoozed,
    snoozedPayloadByKey: nextSnoozedPayloads,
    dismissedByKey: nextDismissed,
    recentlySeenByKey: nextSeen,
    notificationQueue: nextQueue,
    inFlightNotification: nextInFlight,
  });
}
